//  $Id: fmdma.cc,v 1.2 2001/07/13 15:15:06 nishi Exp $
//
//  Copyright (C) 2001 Shouhei Nishi.
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA



#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS bx_dma.

#define DMA_MODE_DEMAND  0
#define DMA_MODE_SINGLE  1
#define DMA_MODE_BLOCK   2
#define DMA_MODE_CASCADE 3



bx_dma_c bx_dma;
#if BX_USE_DMA_SMF
#define this (&bx_dma)
#endif



bx_dma_c::bx_dma_c(void)
{
	setprefix("[DMA ]");
	settype(DMALOG);
	BX_DEBUG(("Init.\n"));
}

bx_dma_c::~bx_dma_c(void)
{
	BX_DEBUG(("Exit.\n"));
}


  void
bx_dma_c::init(bx_devices_c *d)
{
  BX_DMA_THIS devices = d;

  /* 71071 DMA controller */

  // 00A0..00AF
  unsigned i;
  for (i=0x00A0; i<=0x00AF; i++) {
    BX_DMA_THIS devices->register_io_read_handler(this, read_handler,
                                        i, "DMA controller");
    BX_DMA_THIS devices->register_io_write_handler(this, write_handler,
                                        i, "DMA controller");
    }

  // 00B0..00BF
  for (i=0x00B0; i<=0x00BF; i++) {
    BX_DMA_THIS devices->register_io_read_handler(this, read_handler,
                                        i, "extra DMA controller");
    BX_DMA_THIS devices->register_io_write_handler(this, write_handler,
                                        i, "extra DMA controller");
    }

  // all channel masked
  for(i=0;i<8;i++) {
    BX_DMA_THIS s.mask[i] = 1;
  }
}



  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_dma_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_DMA_SMF
  bx_dma_c *class_ptr = (bx_dma_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}

  /* 71071 DMA controller */
  Bit32u
bx_dma_c::read( Bit32u   address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_DMA_SMF
  Bit16u value;

  if(io_len == 1) {
    value = BX_DMA_THIS read_main(address);
  } else if ((address == 0x00a2 ||
	      address == 0x00a4 ||
	      address == 0x00a8) &&
	     io_len == 2) {
    value = ( BX_DMA_THIS read_main(address    )       & 0x00ff) ||
             ((BX_DMA_THIS read_main(address + 1) << 8) & 0xff00);
  } else {      
    BX_PANIC(("dma: io read from address %08x, len=%u\n",
             (unsigned) address, (unsigned) io_len));
    return(0);
  }

  if (bx_dbg.dma) {
    if (io_len == 1) {
      BX_INFO(("dma: read: address=%04x value=%02x\n",
	       (unsigned) address, (unsigned) value));
    } else {
      BX_INFO(("dma: read: address=%04x value=%04x\n",
	       (unsigned) address, (unsigned) value));
    }
  }
  return(value);
}
  Bit32u
bx_dma_c::read_main( Bit32u   address)
{
  Bit8u channeloff;
  Bit8u dmaoff;

  switch(address & 0xf0) {
  case 0xa0:
    channeloff=0;
    dmaoff=0;
    break;
  case 0xb0:
    channeloff=4;
    dmaoff=1;
    break;
  default:
    goto error;
  }

  switch (address&0x0f) {

    case 0x04:
      if(BX_DMA_THIS s.base[dmaoff]) {
	return( BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
		.base_address        & 0xff);
      } else {
	return( BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
		.current_address        & 0xff);
      }
    case 0x05:
      if(BX_DMA_THIS s.base[dmaoff]) {
	return((BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
		.base_address >>  8) & 0xff);
      } else {
	return((BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
		.current_address >> 8 ) & 0xff);
      }
    case 0x06:
      if(BX_DMA_THIS s.base[dmaoff]) {
	return((BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
		.base_address >> 16) & 0xff);
      } else {
	return((BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
		.current_address >> 16) & 0xff);
      }
    case 0x07:
      return((BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	      .extra_address >> 24) & 0xff);

    case 0x0f:
      return((BX_DMA_THIS s.mask[channeloff + 0] ? 0x01 : 0x00) |
	     (BX_DMA_THIS s.mask[channeloff + 1] ? 0x02 : 0x00) |
	     (BX_DMA_THIS s.mask[channeloff + 2] ? 0x04 : 0x00) |
	     (BX_DMA_THIS s.mask[channeloff + 3] ? 0x08 : 0x00));

    default:
    error:
      BX_PANIC(("dma: read: unsupported address=%04x\n", (unsigned) address));
      return(0);
    }
}


  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_dma_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_DMA_SMF
  bx_dma_c *class_ptr = (bx_dma_c *) this_ptr;

  class_ptr->write(address, value, io_len);
}


  /* 71071 DMA controller */
  void
bx_dma_c::write(Bit32u   address, Bit32u   value, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_DMA_SMF

  if (bx_dbg.dma) {
    if (io_len == 1) {
      BX_INFO(("dma: write: address=%04x value=%02x\n",
	       (unsigned) address, (unsigned) value));
    } else {
      BX_INFO(("dma: write: address=%04x value=%04x\n",
	       (unsigned) address, (unsigned) value));
    }
  }

  if(io_len == 1) {
    BX_DMA_THIS write_main(address, value);
  } else if ((address == 0x00a2 ||
	      address == 0x00a4 ||
	      address == 0x00a8) &&
	     io_len == 2) {
    BX_DMA_THIS write_main(address    , ( value       & 0xff));
    BX_DMA_THIS write_main(address + 1, ((value >> 8) & 0xff));
  } else {      
    BX_PANIC(("dma: io write to address %08x, len=%u\n",
	      (unsigned) address, (unsigned) io_len));
  }
}

  void
bx_dma_c::write_main(Bit32u   address, Bit32u   value)
{
  Bit8u channeloff;
  Bit8u dmaoff;
  int i;

  switch(address & 0xf0) {
  case 0xa0:
    channeloff=0;
    dmaoff=0;
    break;
  case 0xb0:
    channeloff=4;
    dmaoff=1;
    break;
  default:
    goto error;
  }

  switch (address&0x0f) {
    case 0x00:
      if((value & 0x02) == 0) goto error;
      if((value & 0x01) != 0) {
	// RESET
	BX_DMA_THIS s.base[dmaoff]                                  = 0;
	BX_DMA_THIS s.channel[dmaoff]                               = 0;
	for(i=0;i<4;i++) {
	  BX_DMA_THIS s.mask[channeloff + i]                        = 1;
	  BX_DMA_THIS s.RQ[channeloff + i]                          = 0;
	  BX_DMA_THIS s.TC[channeloff + i]                          = 0;
	  BX_DMA_THIS s.SRQ[channeloff + i]                         = 0;
	  BX_DMA_THIS s.chan[channeloff + i].enable                 = 0;
	  BX_DMA_THIS s.chan[channeloff + i].mode.mode_type         = 0;
	  BX_DMA_THIS s.chan[channeloff + i].mode.address_decrement = 0;
	  BX_DMA_THIS s.chan[channeloff + i].mode.autoinit_enable   = 0;
	  BX_DMA_THIS s.chan[channeloff + i].mode.transfer_type     = 0;
	  BX_DMA_THIS s.chan[channeloff + i].mode.word_transfer     = 0;
	  BX_DMA_THIS s.chan[channeloff + i].base_address           = 0;
	  BX_DMA_THIS s.chan[channeloff + i].current_address        = 0;
	  BX_DMA_THIS s.chan[channeloff + i].extra_address          = 0;
	  BX_DMA_THIS s.chan[channeloff + i].base_count             = 0;
	  BX_DMA_THIS s.chan[channeloff + i].current_count          = 0;
	}
      }
      return;
      
    case 0x01:
      BX_DMA_THIS s.base[dmaoff]    = (value & 0x04) != 0;
      BX_DMA_THIS s.channel[dmaoff] = (value & 0x03);
      return;

    case 0x02:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.base_count = 
	(BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	 .base_count & 0xff00) | ( value       & 0x00ff);
      if(!BX_DMA_THIS s.base[dmaoff]) {
	BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	  .current_count = 
	  (BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	   .current_count & 0xff00) | ( value       & 0x00ff);
	}
      return;
    case 0x03:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.base_count = 
	(BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	 .base_count & 0x00ff) | ((value << 8) & 0xff00);
      if(!BX_DMA_THIS s.base[dmaoff]) {
	BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	  .current_count = 
	  (BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	   .current_count & 0x00ff) | ((value << 8) & 0xff00);
	}
      return;

    case 0x04:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.base_address = 
	(BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	 .base_address & 0x00ffff00) | ( value        & 0x000000ff);
      if(!BX_DMA_THIS s.base[dmaoff]) {
	BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	  .current_address = 
	  (BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	   .current_address & 0x00ffff00) | ( value        & 0x000000ff);
	}
      return;
    case 0x05:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.base_address = 
	(BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	 .base_address & 0x00ff00ff) | ((value <<  8) & 0x0000ff00);
      if(!BX_DMA_THIS s.base[dmaoff]) {
	BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	  .current_address = 
	  (BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	   .current_address & 0x00ff00ff) | ((value <<  8) & 0x0000ff00);
	}
      return;
    case 0x06:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.base_address = 
	(BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	 .base_address & 0x0000ffff) | ((value << 16) & 0x00ff0000);
      if(!BX_DMA_THIS s.base[dmaoff]) {
	BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	  .current_address = 
	  (BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	   .current_address & 0x0000ffff) | ((value << 16) & 0x00ff0000);
	}
      return;
    case 0x07:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.extra_address = (value << 24) & 0xff000000;
      return;

    case 0x08:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]].enable
	= (value & 0x04) == 0;
      if((value&0xfb)!=0x20) goto error;
      //if(value!=0x20) goto error;
      return;

    case 0x09:
      if(value != 0x00) goto error;
      return;

    case 0x0a:
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.mode.mode_type         = (value & 0xc0) >> 6;
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.mode.address_decrement = (value & 0x20) != 0;
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.mode.autoinit_enable   = (value & 0x10) != 0;
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.mode.transfer_type     = (value & 0x0c) >> 2;
      BX_DMA_THIS s.chan[channeloff + BX_DMA_THIS s.channel[dmaoff]]
	.mode.word_transfer     = (value & 0x01) != 0;
      return;

    case 0x0f:
      BX_DMA_THIS s.mask[channeloff + 0] = (value & 0x01) != 0;
      BX_DMA_THIS s.mask[channeloff + 1] = (value & 0x02) != 0;
      BX_DMA_THIS s.mask[channeloff + 2] = (value & 0x04) != 0;
      BX_DMA_THIS s.mask[channeloff + 3] = (value & 0x08) != 0;
      return;

    default:
    error:
      BX_PANIC(("DMA: unsupported write: %04xh = %02xh\n",
        (unsigned) address, (unsigned) value));
    }
}

  void
bx_dma_c::DRQ(unsigned channel, Boolean val)
{
  Bit32u dma_base, dma_roof;

#if 0
  if ( channel != 2 )
    BX_PANIC(("bx_dma_c::DRQ(): channel %d != 2\n",
	      channel));
#endif
  if (!val) {
    //BX_INFO(("bx_dma_c::DRQ(): val == 0\n");
    // clear bit in status reg
    // deassert HRQ if not pending DRQ's ?
    // etc.
    BX_DMA_THIS s.RQ[channel] = 0;
    return;
    }

#if 0
  BX_INFO(("BX_DMA_THIS s.mask[2]: %02x\n", (unsigned) BX_DMA_THIS s.mask[2]));
  BX_INFO(("BX_DMA_THIS s.flip_flop: %u\n", (unsigned) BX_DMA_THIS s.flip_flop));
  BX_INFO(("BX_DMA_THIS s.status_reg: %02x\n", (unsigned) BX_DMA_THIS s.status_reg));
  BX_INFO(("mode_type: %02x\n", (unsigned) BX_DMA_THIS s.chan[channel].mode.mode_type));
  BX_INFO(("address_decrement: %02x\n", (unsigned) BX_DMA_THIS s.chan[channel].mode.address_decrement));
  BX_INFO(("autoinit_enable: %02x\n", (unsigned) BX_DMA_THIS s.chan[channel].mode.autoinit_enable));
  BX_INFO(("transfer_type: %02x\n", (unsigned) BX_DMA_THIS s.chan[channel].mode.transfer_type));
  BX_INFO((".base_address: %04x\n", (unsigned) BX_DMA_THIS s.chan[channel].base_address));
  BX_INFO((".current_address: %04x\n", (unsigned) BX_DMA_THIS s.chan[channel].current_address));
  BX_INFO((".base_count: %04x\n", (unsigned) BX_DMA_THIS s.chan[channel].base_count));
  BX_INFO((".current_count: %04x\n", (unsigned) BX_DMA_THIS s.chan[channel].current_count));
  BX_INFO((".page_reg: %02x\n", (unsigned) BX_DMA_THIS s.chan[channel].page_reg));
#endif

  BX_DMA_THIS s.RQ[channel] = 1;

  //  if (BX_DMA_THIS s.mask[channel])
  //    BX_PANIC(("bx_dma_c::DRQ(): BX_DMA_THIS s.mask[] is set\n"));


  if ( (BX_DMA_THIS s.chan[channel].mode.mode_type != DMA_MODE_SINGLE) &&
       (BX_DMA_THIS s.chan[channel].mode.mode_type != DMA_MODE_DEMAND) )
    BX_PANIC(("bx_dma_c::DRQ: mode_type(%02x) not handled\n",
	      (unsigned) BX_DMA_THIS s.chan[channel].mode.mode_type));
  if (BX_DMA_THIS s.chan[channel].mode.address_decrement != 0)
    BX_PANIC(("bx_dma_c::DRQ: address_decrement != 0\n"));
  //if (BX_DMA_THIS s.chan[channel].mode.autoinit_enable != 0)
  //  BX_PANIC(("bx_dma_c::DRQ: autoinit_enable != 0\n"));

  dma_base = BX_DMA_THIS s.chan[channel].extra_address | BX_DMA_THIS s.chan[channel].base_address;
  if(BX_DMA_THIS s.chan[channel].mode.word_transfer) {
    BX_PANIC(("dma: word transfer not implemented."));
    dma_roof = dma_base + BX_DMA_THIS s.chan[channel].base_count * 2 + 1;
  } else {
    dma_roof = dma_base + BX_DMA_THIS s.chan[channel].base_count;
  }
  if ( (dma_base & 0xff000000) != (dma_roof & 0xff000000) ) {
BX_INFO(("dma_base = %08x\n", (unsigned) dma_base));
BX_INFO(("dma_base_count = %08x\n", (unsigned) BX_DMA_THIS s.chan[channel].base_count));
BX_INFO(("dma_roof = %08x\n", (unsigned) dma_roof));
    BX_PANIC(("dma: DMA request outside 16M boundary\n"));
    }


  //BX_INFO(("DRQ set up for single mode, increment, auto-init disabled, write\n"));
  // should check mask register VS DREQ's in status register here?
  // assert Hold ReQuest line to CPU
  bx_pc_system.set_HRQ(1);
}

  void
bx_dma_c::raise_HLDA(bx_pc_system_c *pc_sys)
{
  unsigned channel;
  Bit32u phy_addr;
  Boolean count_expired = 0;

  // find highest priority channel
  for (channel=0; channel<8; channel++) {
    if ( BX_DMA_THIS s.RQ[channel] && ! BX_DMA_THIS s.mask[channel] &&
	 BX_DMA_THIS s.chan[channel].enable) {
      break;
      }
    }
  if (channel >= 8) {
	// don't panic, just wait till they're unmasked
    //    BX_PANIC(("hlda: no unmasked requests\n"));
    return;
    }

  //BX_INFO(("hlda: OK in response to DRQ(%u)\n", (unsigned) channel));
  phy_addr = BX_DMA_THIS s.chan[channel].extra_address |
             BX_DMA_THIS s.chan[channel].current_address;

  bx_pc_system.set_DACK(channel, 1);
  // check for expiration of count, so we can signal TC and DACK(n)
  // at the same time.
  if (BX_DMA_THIS s.chan[channel].mode.address_decrement==0) {
    // address increment
    BX_DMA_THIS s.chan[channel].current_address++;
    BX_DMA_THIS s.chan[channel].current_count--;
    if (BX_DMA_THIS s.chan[channel].current_count == 0xffff) 
      if (BX_DMA_THIS s.chan[channel].mode.autoinit_enable == 0) {
	// count expired, done with transfer
	// assert TC, deassert HRQ & DACK(n) lines
	BX_DMA_THIS s.TC[channel] = 1; // hold TC in status reg
	bx_pc_system.set_TC(1);
	count_expired = 1;
      } else {
	// count expired, but in autoinit mode
	// reload count and base address
	BX_DMA_THIS s.chan[channel].current_address = 
	  BX_DMA_THIS s.chan[channel].base_address;
	BX_DMA_THIS s.chan[channel].current_count =
	  BX_DMA_THIS s.chan[channel].base_count;
      }
    
    }
  else {
    // address decrement
    BX_PANIC(("hlda: decrement not implemented\n"));
    }

  if (BX_DMA_THIS s.chan[channel].mode.transfer_type == 1) { // write
    // xfer from I/O to Memory
    pc_sys->dma_write8(phy_addr, channel);
    }
  else if (BX_DMA_THIS s.chan[channel].mode.transfer_type == 2) { // read
    // xfer from Memory to I/O
    pc_sys->dma_read8(phy_addr, channel);
    }
  else {
    BX_PANIC(("hlda: transfer_type of %u not handled\n",
      (unsigned) BX_DMA_THIS s.chan[channel].mode.transfer_type));
    }

  if (count_expired) {
    bx_pc_system.set_TC(0);            // clear TC, adapter card already notified
    bx_pc_system.set_HRQ(0);           // clear HRQ to CPU
    bx_pc_system.set_DACK(channel, 0); // clear DACK to adapter card
    }
}
#endif
